package csokicraft.util.signlib;

import java.io.*;
import java.security.*;
import java.security.spec.*;

import org.apache.commons.io.IOUtils;

/** Implementation of SignLib functions. Basically a wrapper for java.security features
  * @author CsokiCraft */
public class SignLibImpl{
	public static final String  DEFAULT_GEN_ALGO="RSA",
								DEFAULT_SIG_ALGO="SHA1WithRSA";
	
	/** Generates a 2k {@link KeyPair} with the given algo */
	public static KeyPair createPair(final String algo) throws NoSuchAlgorithmException{
		KeyPairGenerator keygen=KeyPairGenerator.getInstance(algo);
		keygen.initialize(2048);
		return keygen.generateKeyPair();
	}
	
	/** Generates a 2k {@link KeyPair} with the default algo */
	public static KeyPair createPair() throws NoSuchAlgorithmException{
		return createPair(DEFAULT_GEN_ALGO);
	}
	
	/** Writes the {@link KeyPair} to the two streams and closes them. <br>
	  * I cannot be held responsible if you've written your private key to the wrong stream. */
	public static void writeKeyPair(KeyPair pair, OutputStream pubkey, OutputStream privkey) throws IOException{
		pubkey.write(pair.getPublic().getEncoded());
		pubkey.close();
		privkey.write(pair.getPrivate().getEncoded());
		privkey.close();
	}
	
	public static PrivateKey readPrivKey(InputStream in) throws IOException, GeneralSecurityException{
		ByteArrayOutputStream privBytes=new ByteArrayOutputStream();
		IOUtils.copy(in, privBytes);
		
		KeyFactory factory=KeyFactory.getInstance(DEFAULT_GEN_ALGO);
		return factory.generatePrivate(new PKCS8EncodedKeySpec(privBytes.toByteArray()));
		
	}
	
	public static PublicKey readPubKey(InputStream in) throws IOException, GeneralSecurityException{
		ByteArrayOutputStream pubBytes=new ByteArrayOutputStream();
		IOUtils.copy(in, pubBytes);

		KeyFactory factory=KeyFactory.getInstance(DEFAULT_GEN_ALGO);
		return factory.generatePublic(new X509EncodedKeySpec(pubBytes.toByteArray()));
	}
	
	/** Signs 'data' and writes the signature to 'sig'
	  * @param algo The signing algorithm */
	public static void sign(InputStream data, OutputStream sig, PrivateKey privkey, final String algo) throws IOException, GeneralSecurityException{
		ByteArrayOutputStream dataBytes=new ByteArrayOutputStream();
		IOUtils.copy(data, dataBytes);
		
		Signature sg=Signature.getInstance(algo);
		sg.initSign(privkey);
		sg.update(dataBytes.toByteArray());
		sig.write(sg.sign());
	}
	
	/** Signs 'data' and writes the signature to 'sig' */
	public static void sign(InputStream data, OutputStream sig, PrivateKey privkey) throws IOException, GeneralSecurityException{
		sign(data, sig, privkey, DEFAULT_SIG_ALGO);
	}
	
	/** Verifies 'data' using the signature 'sig'
	  * @param algo The signing algorithm */
	public static boolean verify(InputStream data, InputStream sig, PublicKey pubkey, final String algo) throws IOException, GeneralSecurityException{
		ByteArrayOutputStream dataBytes=new ByteArrayOutputStream();
		IOUtils.copy(data, dataBytes);
		ByteArrayOutputStream sigBytes=new ByteArrayOutputStream();
		IOUtils.copy(sig, sigBytes);
		
		Signature sg=Signature.getInstance(algo);
		sg.initVerify(pubkey);
		sg.update(dataBytes.toByteArray());
		return sg.verify(sigBytes.toByteArray());
	}
	
	/** Verifies 'data' using the signature 'sig' */
	public static boolean verify(InputStream data, InputStream sig, PublicKey pubkey) throws IOException, GeneralSecurityException{
		return verify(data, sig, pubkey, DEFAULT_SIG_ALGO);
	}
}
