package csokicraft.util.signlib;

import java.io.*;
import java.nio.*;
import java.security.*;
import java.util.*;

import csokicraft.util.IndexableTable;

/** The abstract SignLib host (server). Capable of sending HIs, receiving SUPs and verifying them.
  * @author CsokiCraft */
public abstract class SignLibHost{
	public static final int PKID_HI=1, PKID_SUP=2;
	
	protected Map<Integer, ByteBuffer> data;
	protected Random rng;
	protected PublicKey pubkey;
	
	public SignLibHost(PublicKey key, Random rand){
		data=new IndexableTable<>();
		pubkey=key;
		rng=rand;
	}
	
	public SignLibHost(PublicKey key){
		this(key, new Random());
	}

	/** Handles the connection event */
	public void onConnect(OutputStream out) throws IOException{
		byte[] bData=new byte[rng.nextInt(255)];
		rng.nextBytes(bData);
		int id=data.size();
		
		sendHi(out, id, bData);
	}
	
	/** Sends a Host-originated Inquiry */
	protected void sendHi(OutputStream out, int id, byte[] newData) throws IOException{
		out.write(PKID_HI);
		out.write(id);
		out.write(newData.length);
		out.write(newData.length>>8);
		out.write(newData);
		data.put(id, ByteBuffer.wrap(newData));
	}

	/** Handles inbound data */
	public void onInbound(InputStream in) throws IOException{
		int pkid=in.read();
		switch (pkid){
		case PKID_SUP:
			try {
				processSup(in);
			} catch (GeneralSecurityException e){
				throw new IOException("Couldn't verify! "+e.getMessage(), e);
			}
			break;

		default:
			System.err.println("Invalid packet discriminator: "+pkid+"!");
			break;
		}
	}
	
	/** Processes an inbound Signed User-originated Packet */
	protected void processSup(InputStream in) throws IOException, GeneralSecurityException{
		int id=in.read();
		byte[] bData=data.get(id).array();
		byte[] bSig=new byte[256];
		in.read(bSig);
		int cmdLenLo=in.read();
		int cmdLenHi=in.read();
		int cmdLen=(cmdLenHi<<8)|cmdLenLo;
		byte[] bCmd=new byte[cmdLen];
		in.read(bCmd);
		
		byte[] bPayload=new byte[bData.length+bCmd.length];
		System.arraycopy(bData, 0, bPayload, 0, bData.length);
		System.arraycopy(bCmd, 0, bPayload, bData.length, bCmd.length);
		
		ByteArrayInputStream inData=new ByteArrayInputStream(bPayload),
							 inSig =new ByteArrayInputStream(bSig);
		
		boolean b=SignLibImpl.verify(inData, inSig, pubkey);
		if(b)
			processCmd(bCmd);
		else
			onRefuse(bCmd);
		
		onClose(id);
	}

	/** Override this method for functionality. Called when authentication succeeds */
	protected abstract void processCmd(byte[] bCmd);

	/** Override this method for functionality. Called when authentication fails */
	protected abstract void onRefuse(byte[] bCmd);

	/** Ends the session with the given id. Call this when the connection closes. Called automatically after the request has been processed */
	public void onClose(int id) throws IOException{
		data.remove(id);
	}
}
