package csokicraft.util.model;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;

import javax.vecmath.Point3d;

/** A model loaded from a Wavefront .OBJ file
 * @author CsokiCraft */
public class ObjModel{
	public static Logger log = Logger.getLogger("[CsCr-ObjLoader] ");
	
	public String objName;
	/** Vertex, Normal and Texture coords */
	public List<Point3d> verts, normals, texCoords;
	/** Faces (triangles) */
	public List<ObjFace> faces;

	private List<String> lastGroup;
	
	public ObjModel(String name){
		this();
		objName=name;
	}
	
	public ObjModel(){
		verts=new ArrayList<>();
		normals=new ArrayList<>();
		texCoords=new ArrayList<>();
		
		lastGroup=new ArrayList<>();
		lastGroup.add("default");
	}
	
	public static ObjModel fromStream(BufferedReader in) throws IOException{
		ObjModel model = new ObjModel();
		int lnNo=0;
		while(in.ready()){
			String ln=in.readLine().trim();
			lnNo++;
			if(ln.isEmpty()||ln.startsWith("#")) continue;
			if(ln.startsWith("o")) nameModel(ln.substring(2), model);
			if(ln.startsWith("g")) setGroup(ln, model);
			if(ln.startsWith("s")) log.warning("\"Set smooth shading\" operation is unsupported! at line "+lnNo);
			
			if(ln.startsWith("v")) processVertex(ln, model);
			if(ln.startsWith("vn")) processNormal(ln, model);
			if(ln.startsWith("vt")) processTexCoord(ln, model);
			if(ln.startsWith("vp")) log.warning("\"Vertex param space\" operation is unsupported! at line "+lnNo);
			
			if(ln.startsWith("f")) processFace(ln, model);
			
			if(ln.startsWith("mtllib")) log.warning("\"Load material lib\" operation is unsupported! at line "+lnNo);
			if(ln.startsWith("usemtl")) log.warning("\"Use material\" operation is unsupported! at line "+lnNo);
			
			if(ln.startsWith("call")) log.severe("\"Call object file\" is not supported due to technological constraints and security issues! at line"+lnNo);
		}
		if(model.objName==null){
			log.fine("Successfully loaded model!");
		}else{
			log.fine("Successfully loaded model: "+model.objName);
		}
		return model;
	}

	private static void nameModel(String name, ObjModel model){
		if(model.objName != null)
			log.info("Model \""+model.objName+"\" will be renamed to \""+name+"\".");
		model.objName = name;
	}

	private static void setGroup(String s, ObjModel model){
		String[] data=s.split(" ");
		model.lastGroup=Arrays.asList(data);
	}

	private static void processVertex(String s, ObjModel model){
		model.verts.add(toPoint3d(s.split(" ")));
	}

	private static void processNormal(String s, ObjModel model){
		model.normals.add(toPoint3d(s.split(" ")));
	}

	private static void processTexCoord(String s, ObjModel model){
		model.texCoords.add(toPoint3d(s.split(" ")));
	}

	private static void processFace(String s, ObjModel model){
		ObjFace f=new ObjFace();
		String[] data=s.split(" ");
		String[] firstEl=data[1].split("/"),
				 secondEl=data[2].split("/"),
				 thirdEl=data[3].split("/");
		f.vertIDs=new int[]{Integer.valueOf(firstEl[0]), Integer.valueOf(secondEl[0]), Integer.valueOf(thirdEl[0])};
		if(firstEl.length>1){
			f.texCoordIDs=new int[]{Integer.valueOf(firstEl[1]), Integer.valueOf(secondEl[1]), Integer.valueOf(thirdEl[1])};
			if(firstEl.length>2){
				f.normalIDs=new int[]{Integer.valueOf(firstEl[2]), Integer.valueOf(secondEl[2]), Integer.valueOf(thirdEl[2])};
			}
		}
		f.groups=model.lastGroup;
		model.faces.add(f);
	}

	/** @return A point from the values in the array (minus its first element). Y defaults to 0, Z to 1.*/
	protected static Point3d toPoint3d(String[] data){
		double x=Double.valueOf(data[1]);
		double y;
		if(data.length>2) y=Double.valueOf(data[2]);
		else y=0;
		double z;
		if(data.length>3) z=Double.valueOf(data[3]);
		else z=1;
		return new Point3d(x, y, z);
	}

	public static ObjModel fromStream(InputStream in) throws IOException{
		return fromStream(new BufferedReader(new InputStreamReader(in)));
	}
	
	public static class ObjFace{
		/** Vertex, Normal and Texture IDs */
		public int[] vertIDs, texCoordIDs, normalIDs;
		public List<String> groups;
		
		public ObjFace(){
			vertIDs = new int[3];
			texCoordIDs = new int[3];
			normalIDs = new int[3];
			groups=new ArrayList<>();
		}
	}
}
