package csokicraft.util.eqdf.novacompiler;

import java.util.*;

import csokicraft.util.eqdf.MathContext;
import csokicraft.util.eqdf.compile.*;
import csokicraft.util.eqdf.exceptions.EqdfCompileException;
import csokicraft.util.eqdf.novacompiler.ast.*;
import csokicraft.util.eqdf.novacompiler.ast.impl.*;

/** A more advanced, experimental {@link IEqdfCompiler}<br>
  * It first tokenizes the {@link String} into a token stream, then turns that into
  * a symbol stream, which is used to build the AST (Abstract Syntax Tree).
  * The bytecode is then created from the AST.<br><br>
  * A token stream is a collection of substrings of the statement, in which the
  * original order and quantity of characters is maintained. Tokens also store their "role"
  * in the statement
  * @see SyntaxNode Nodes in AST */
public class NovaEqdfCompiler implements IEqdfCompiler{

	@Override
	public void clean(){
		
	}

	@Override
	public MathContext compile(String s){
		String sides[]=s.split("=");
		String rhs=s, lhs=null;
		if(sides.length==2) {
			lhs=sides[0];
			rhs=sides[1];
		}else if(sides.length>2) {
			throw new EqdfCompileException("Too many '='!");
		}
		SyntaxNode root=getRoot(rhs);
		List<Object> l=astCompile(root);
		if(lhs!=null) {
			if(lhs.startsWith("$")) {
				l.add(EqdfInstructionSet.ST_VAR);
				l.add(lhs.charAt(1));
			}else {
				throw new EqdfCompileException(lhs+" is not a variable!");
			}
		}
		return new MathContext(l);
	}
	
	private SyntaxNode getRoot(String s) {
		Queue<Token> tokenStream=tokenize(s);
		SyntaxNode root=astBuild(tokenStream);
		return root;
	}
	
	/** Tokenizes the {@link String} */
	protected Queue<Token> tokenize(String s){
		Queue<Token> tokenStream=new LinkedList<>();
		/** Stores the last, unprocessed multi-character token */
		StringBuilder sb=new StringBuilder();
		char[] arr=s.toCharArray();
		for(int i=0;i<arr.length;i++){
			switch(arr[i]) {
			case '+':
			case '-':
			case '*':
			case '/':
			case '%':
				flushTokenBuf(sb, tokenStream);
				tokenStream.add(new Token(String.valueOf(arr[i]), EnumTokenType.OPERATOR));
				break;
			case '(':
			case ')':
			case ',':
				flushTokenBuf(sb, tokenStream);
				tokenStream.add(new Token(String.valueOf(arr[i]), EnumTokenType.SYNTAX));
				break;
			case ' ':
				flushTokenBuf(sb, tokenStream);
				break;
			case '$':
				char varName=arr[++i];
				tokenStream.offer(new Token(String.valueOf(varName), EnumTokenType.VARIABLE));
				break;
			default:
				sb.append(arr[i]);
			}
		}
		flushTokenBuf(sb, tokenStream);
		return tokenStream;
	}
	
	private static void flushTokenBuf(StringBuilder buf, Queue<Token> tokenStream) {
		if(buf.length()>0) {
			tokenStream.add(new Token(buf.toString(), EnumTokenType.VALUE));
			buf.delete(0, buf.length());
		}
	}
	
	/** Builds the AST from the token stream */
	protected static SyntaxNode astBuild(Queue<Token> tokenStream){
		SyntaxNode last=null;
		while(!tokenStream.isEmpty()) {
			last=getNextNode(last, tokenStream);
		}
		return last;
	}
	
	/** You better make node of that! */
	protected static SyntaxNode getNextNode(SyntaxNode last, Queue<Token> tokenStream) {
		Token t=tokenStream.poll();
		switch(t.type) {
		case VALUE:
			if(last!=null)
				throw new EqdfCompileException("Invalid token before '"+t.value+"'!");
			return new ValueNode(Double.parseDouble(t.value));
		case VARIABLE:
			if(last!=null)
				throw new EqdfCompileException("Invalid token before '"+t.value+"'!");
			return new VariableNode(t.value.charAt(0));
		case OPERATOR:
			if(last==null) {
				if("-".equals(t.value)) {
					return new UnaryNegationNode(getNextNode(null, tokenStream));
				}else
					throw new EqdfCompileException("'"+t.value+"' can not be used as unary operator!");
			}else {
				return new OperationNode(last, getNextNode(null, tokenStream), t.value.charAt(0));
			}
		case FUNCTION:
			List<SyntaxNode> l=new ArrayList<>();
			int argc=EqdfFunctions.getArgc(t.value);
			if(argc==-1)
				throw new EqdfCompileException("Unknown token: '"+t.value+"', no such function.");
			
			if(tokenStream.isEmpty()) {
				if(argc!=0)
					throw new EqdfCompileException("Unexpected end of stream when fetching parameters for '"+t.value+"'");
			}else if(tokenStream.peek().value.equals("(")) {
				tokenStream.poll(); //Begin bracketed expression by removing opening parenthesis
				BoolWrapper endOfStream=new BoolWrapper(false);
				if(argc==0) {
					if(getNextExpr(tokenStream, endOfStream)!=null)
						throw new EqdfCompileException("'"+t.value+"' takes no arguments!"); //
				}else for(int i=0;i<argc;i++) {
					if(endOfStream.value)
						throw new EqdfCompileException("Too few arguments for function '"+t.value+"'!");
					SyntaxNode exp=getNextExpr(tokenStream, endOfStream);
					if(exp==null)
						throw new EqdfCompileException("Empty parameter for '"+t.value+"'!");
					l.add(exp);
				}
				if(!endOfStream.value)
					throw new EqdfCompileException("Too many arguments for function '"+t.value+"'!");
			}else {
				if(argc>1)
					throw new EqdfCompileException("Missing parentheses in function call '"+t.value+"'!");
				else if(argc==1)
					l.add(getNextNode(null, tokenStream));
			}
			
			return new FunctionNode(t.value, l);
		case SYNTAX:
			if("(".equals(t.value)) {
				return getNextExpr(tokenStream, null);
			}
		}
		throw new EqdfCompileException("Could not translate token '"+t.value+"'!");
	}
	
	/** Get the next expression
	  * @param tokenStream the token stream
	  * @param endOfStream a flag, which will be set to true if the expression was terminated by ')', false if by ',' */
	protected static SyntaxNode getNextExpr(Queue<Token> tokenStream, BoolWrapper endOfStream) {
		Queue<Token> subStream=new LinkedList<>();
		int depth=1;
		sub:while(depth!=0) {
			Token tmp=tokenStream.poll();
			if(tmp.type==EnumTokenType.SYNTAX) {
				if("(".equals(tmp.value)) {
					depth++;
				}else if(")".equals(tmp.value)) {
					depth--;
					if(depth==0) {
						if(endOfStream!=null)
							endOfStream.value=true;
						break sub;
					}
				}else if(",".equals(tmp.value)&&depth==1) {
					if(endOfStream!=null)
						endOfStream.value=false;
					break sub;
				}
			}
			subStream.offer(tmp);
		}
		return astBuild(subStream);
	}

	/** Compiles the AST into bytecode */
	protected static List<Object> astCompile(SyntaxNode node){
		List<Object> l=new LinkedList<>();
		node.addInstructions(l);
		return l;
	}
	
	protected static class Token{
		public final String value;
		public final EnumTokenType type;
		
		public Token(String s, EnumTokenType t){
			value=s;
			if(t==EnumTokenType.VALUE) {
				EnumTokenType tmp;
				try {
					Double.parseDouble(value);
					tmp=t;
				}catch(NumberFormatException ex) {
					tmp=EnumTokenType.FUNCTION;
				}
				type=tmp;
			}else {
				type=t;
			}
		}
	}
	
	protected static enum EnumTokenType{
		VALUE, OPERATOR, FUNCTION, VARIABLE, SYNTAX;
	}
	
	protected static class BoolWrapper{
		public boolean value;
		
		public BoolWrapper(boolean b){
			value=b;
		}
	}
}
