package csokicraft.util.eqdf;

import java.util.*;

import csokicraft.util.eqdf.compile.EqdfFunctions;
import static csokicraft.util.eqdf.compile.EqdfInstructionSet.*;

/** A class for executing EQDF instructions on a set of variables.
  * Also doubles as a miniature math utility collection.
  * @author CsokiCraft */
public class MathContext{
	protected Map<Character, Double> vars;
	protected List<Object> instructions;
	
	public MathContext(List<Object> l){
		vars=new HashMap<>();
		instructions=l;
	}
	
	public void setVar(char var, double val){
		vars.put(var, val);
	}
	
	public double getVar(char var){
		return vars.get(var);
	}
	
	public List<Double> exec(){
		List<Double> l = new ArrayList<Double>();
		if(instructions == null) return l;
		Stack<Double> stack = new Stack<Double>();
		Map<Integer, Double> brackets = new HashMap<Integer, Double>();
		for(int idx=0;idx<instructions.size();idx++){
			Object obj = instructions.get(idx);
			if(obj.equals(LD_VAR)){
				Object key = instructions.get(++idx);
				Double var = vars.get(key);
				stack.push(var);
			}else if(obj.equals(LD_BRA)){
				Object key = instructions.get(++idx);
				Double var = brackets.get(key);
				stack.push(var);
			}else if(obj.equals(ST_VAR)){
				Object key = instructions.get(++idx);
				Double var = stack.pop();
				vars.put((Character) key, var);
			}else if(obj.equals(ST_BRA)){
				Object key = instructions.get(++idx);
				Double var = stack.pop();
				brackets.put((Integer) key, var);
			}else if(obj.equals(CONST)){
				Double var = (Double) instructions.get(++idx);
				stack.push(var);
			}else if(obj.equals(OP_ADD)){
				stack.push(stack.pop()+stack.pop());
			}else if(obj.equals(OP_MUL)){
				stack.push(stack.pop()*stack.pop());
			}else if(obj.equals(OP_SBT)){
				stack.push(-stack.pop()+stack.pop());
			}else if(obj.equals(OP_DIV)){
				double  b = stack.pop(),
						a = stack.pop();
				stack.push(a/b);
			}else if(obj.equals(OP_DINT)){
				long b = floorLong(stack.pop()),
					 a = floorLong(stack.pop()),
					 res = a/b;
				stack.push(Long.valueOf(res).doubleValue());
			}else if(obj.equals(OP_MOD)){
				double  b = stack.pop(),
						a = stack.pop();
				stack.push(a%b);
			}else if(obj.equals(CALLF)){
				String funName = (String) instructions.get(++idx);
				EqdfFunctions.callF(funName, stack);
			}
		}
		
		while(!stack.isEmpty()) l.add(stack.pop());
		return l;
	}
	
	/** Import variables from the given {@link MathContext} <br>
	  * The values of the parameter overwrite pre-existing ones. <br>
	  * @param ctx the source. If null, it is ignored */
	public void importVars(MathContext ctx){
		if(ctx == null || ctx.vars.isEmpty()) return;
		for(Map.Entry<Character, Double> e:ctx.vars.entrySet()){
			vars.put(e.getKey(), e.getValue());
		}
	}
	
	public static long floorLong(double d){
		return Math.round(Math.floor(d));
	}
	
	public static int floorInt(double d){
		return Double.valueOf(Math.floor(d)).intValue();
	}
}
